'==============================================================================
'
' PE32.VBS Functions List:
'
'==============================================================================
'
' General usage list (often assigned to a key)
'

' Align_Col
'   Align blank / non-blank transition at cursor (columns past column
'   being aligned are shifted in unison).

' Align_Word
'   Search for all occurrences of current word and try to align them.

' Clear_Books(Glb)
'   Clear all bookmarks

' Close_All
'   Save changed files and close all files.

' Compare
'   Visual flash compare current and next file.

' Copy_Mark
'   Copy mark to cursor position and move highlighting to new position.

' End_All
'   Move cursor to end of line on all files.

' Ex_Commands (FileSpec)
'   Execute a file of PE32 and/or VBS commands.

' Goto_Book(Direction, Glb)
'   Find next bookmark in specified Direction
'   Look in current file unless Glb specified direction to search in multiple files

' GotoDiff
'   Goto next line which is different between current and next file.

' GotoErr
'   Parse C error line.

' Home_All
'   Move cursor to beginning of line on all files.

' Left_Justify
'   Place cursor at left edge of field to be left justified.

' Move_Mark
'   Move mark to cursor position and move highlighting to new position.

' Number_Mark
'   Add numbers to the line or block mark

' Prompt_Global (Commands)
'   Prompt for a string of PE32 commands and then execute them on all open files.

' Read_Only
'   Open current file in read only mode.

' Reflow_Mult(Guess_Para)
'   Reflow the area covered by a mark.

' Rem_Book(Start)
'   Remove last bookmark at current location beyond Start

' RemAll
'   Comment out (add // rem to start of line) all of file.

' RemFun
'   From current position until matching C block depth
'   comment out (add // rem to start of line) all source lines.

' RemMark
'   Comment out (add // rem to start of line) marked lines.

' RemWord
'   Comment out (with block comments) a single word.

' Right_Justify
'   Place cursor at right edge of field to be right justified.

' Save_All
'   Save all files.

' Save_Changed
'   Save all changed files.

' Set_Book(Start)
'   Set the next available bookmark beyond Start and all bookmarks at current location

' Set_Margins
'   Set margins to edges of block mark.

' Tab_N_Left(Columns)
'   Tab left marked lines by passed number of columns.

' Tab_N_Right(Columns)
'   Tab right marked lines in current file by passed number of columns.

' UnNumber_Mark
'   Remove numbers from the line or block mark


'==============================================================================
'
' Utilities generally called from other scripts
'

' Flash (Wait_Time, Num_Flash)
'   Visual flash 2 files.

' Force_Refresh
'   Force PE32 to update the screen.

' Letter = Get_Book(Direction)
'   Search in current file in the specified direction from cursor for a bookmark
'   Return last bookmark letter found or zero if none is found

' Get_Mark_Size(CommandM, StartRow, EndRow, StartCol, EndCol)
'   Return mark size and whether it is on the command line.

' Get_Screen_Line
'   Fetch line of screen that has cursor.

' Is_Blanks(String)
'   Returns true if String consists of all blanks.

' Mark_Type
'   Check current area (current file; data area or command line) for mark type.

' Letter = New_Book(Start)
'   Get the next available bookmark beyond Start or zero if none are available

' Rest_Screen_State(CommandT, CommandC, DataR, DataC, SLine)
'   Restore state of display.

' Save_Screen_State(CommandT, CommandC, DataR, DataC, SLine)
'   Get current state of display.

' Set_Screen_Line(Line)
'   Restore cursor to passed line of screen.


'==============================================================================
'
' Documentation and inquiry routines
'

' Key10
'   Capture key code for next keystroke as text.

' KeyCode
'   Capture key code for next keystroke as text.

'==============================================================================
'==============================================================================
'
' General usage scripts
'

'
'   For jagged columns:  align blank / non-blank transition at cursor (columns
'      past column being aligned are shifted in unison).
'
'   Align column to cursor:  if over text add blanks Else remove blanks
'
'   Apply to rest of file if no marked lines Else apply to rest of mark
'
Function Align_Col

  ' Get state
  row1  = pe32.Row                      ' Position
  col1  = pe32.Col
  mode1 = pe32.InsertMode               '  and mode

  pe32.InsertMode = 1                   ' Set to insert

  lines1 = pe32.Lines - row1            ' Get lines and current line in File1

  ' Restrict loop to marked lines
  If pe32.MarkType > 0 And pe32.MarkType < 4 Then

    pe32.Command "[begin mark]"         ' Goto start of mark
    row2 = pe32.Row

    If (row2 < row1) Then row2 = row1   ' Use lower of row1 and row2

    pe32.Command "[end mark]"           ' Goto end of mark
    lines1 = pe32.Row - row2            ' Shorten loop to end of mark

    pe32.Row = row2                     ' Restore start position

  End If

  ' Loop over current and rest of lines
  For i=1 To lines1+1                   ' Loop over rest of lines in file

    str = pe32.Line

    char = Mid(str, col1-1, 1)          ' Extract align character

    If char = " " Then

      ' Count deletable blanks
      For j=col1 To Len(str)

        char = Mid(str, j, 1)           ' Extract j'th character

        If char <> " " Then
          Exit For
        End If

      Next

      ' Remove deletable blanks
      pe32.Line = Left(str, col1-1) & Mid(str, j)
    Else

      ' Count blanks to insert
      For j=col1-1 To 1 step -1

        char = Mid(str, j, 1)           ' Extract j'th character

        If char = " " Then
          Exit For
        End If

      Next

      ' Add blanks
      pe32.Col = j

      For k=1 To col1-j-1

        pe32.Insert = " "               ' Align word

      Next

    End If

    pe32.Row = pe32.Row + 1             ' Get next line
  Next                                  ' Repeat loop

  ' Restore state
  pe32.Row        = row1                ' Restore position
  pe32.Col        = col1
  pe32.InsertMode = mode1               '  and mode

End Function


'
'   Search for all occurrences of current word and try to align them
'      Note:  will use cursor position to align (even if not at start of word)
'
Function Align_Word


  Dim regEx, Matches
  Set regEx = New RegExp                ' Use regular expressions
  regEx.IgnoreCase = False
  regEx.Global = False


  ' Get state
  str1  = pe32.Word                     ' Get current word
  row1  = pe32.Row                      '  and position
  col1  = pe32.Col
  mode1 = pe32.InsertMode               '  and mode

  pe32.InsertMode = 1                   ' Set to insert

  lines1 = pe32.Lines - row1            ' Get lines and current line in File1

  For i=1 To lines1                     ' Loop over rest of lines in file

    pe32.Row = pe32.Row + 1             ' Get next line
    str      = pe32.Line

    ' Find match
    regEx.Pattern = str1                ' Set pattern
    Set Matches   = regEx.Execute(str)  ' Execute search

    If Matches.count > 0 Then           ' If found

      For each Match in Matches         ' Load match information
        Exit For
      Next

      find1    = Match.FirstIndex+1     ' Goto that column
      pe32.Col = find1

      If find1 < col1 Then              ' Before alignment word?

        ' Add blanks
        For j=1 To col1-find1

          pe32.Insert = " "             ' Align word

        Next

      ElseIf find1 > col1 Then          ' After alignment word?


        ' Count deletable blanks
        For j=find1-1 To col1 step -1

          char = Mid(str, j, 1)         ' Extract j'th character

          If char <> " " Then
           Exit For
          End If

        Next

        ' Remove deletable blanks
        pe32.Line = Left(str, j) & Mid(str, find1)

      End If

    End If

  Next                                  ' Repeat loop

  ' Restore state
  pe32.Row        = row1                ' Restore position
  pe32.Col        = col1
  pe32.InsertMode = mode1               '  and mode

End Function


'
'   Clear all bookmarks
'
Function Clear_Books(Glb)

  Dim FirstFile

  FirstFile = pe32.Filename               ' Save starting file
  '
  ' Loop over files
  '
  Do
    CommandT = pe32.CommandMode           ' Save command mode

    pe32.command "[cursor data]"

    ' Loop over bookmarks "A" to "Z"
    For i = 97 To 122
      pe32.command "[bookmark remove '" & Chr(i) & "']" ' Clear the bookmark
    Next

    pe32.CommandMode = CommandT           ' Restore command mode

    If Glb <> 1 Then Exit Do              ' Done if not global

    pe32.Command "e"                      ' Next file

  Loop Until pe32.Filename = FirstFile    ' Exit when back to first file

End Function


'
'   Save and close all files
'
'   Works well to exit a workspace too
'
Function Close_All

  Call Save_Changed()

  pe32.Command "[quit all]"

End Function


'
'   Visual flash compare current and next file.
'
'   Cursor movement keys scroll both files
'      PgUp, PgDn, Home, End, Left, Right
'
'   Esc exits compare
'
'   + goes to next difference
'
'   Ins scrolls first file up   (it has less lines)
'   Del scrolls first file down (it has more lines)
'
'   Shift cursor keys just move File1
'   Ctrl  cursor keys just move File2
'
Function Compare

  Dim Refresh

  Refresh = pe32.Get("vbsrefresh")      ' Save VBS refresh state

  pe32.command "set vbsrefresh on"

' wait_time=250                         ' Milleseconds to wait
  wait_time=50000                       ' Burn counter:  adjust for your
                                        '     computers speed
  num_flash=5

  exitloop=0
  Do

    key   = pe32.Key                    ' Get command
    state = pe32.Shift                  ' Get shift status for last key

    shift   = (state And 16)
    control = (state And 12)

    If (shift > 0) And (control > 0) Then
      MsgBox "state=" & state & " control=" & control
    End If

    Select Case key
      Case 27                           ' Esc
        exitloop=1

      Case &H2200                       ' PgDn
        pe32.Command "[page down]"
        pe32.Command "e"
        pe32.Command "[page down]"
        Call Flash(wait_time, num_flash)

      Case &H2100                       ' PgUp
        pe32.Command "[page up]"
        pe32.Command "e"
        pe32.Command "[page up]"
        Call Flash(wait_time, num_flash)

      Case &H2600                       ' Up
        If control = 0 Then             ' If Not Either_Ctl_Key_Down
          pe32.Command "[scrolldown]"
        End If

        pe32.Command "e"

        If shift = 0 Then               ' If shift = 0_Key_Down
          pe32.Command "[scrolldown]"
        End If

        Call Flash(wait_time, num_flash)

      Case &H2800                       ' Dn
        If control = 0 Then             ' If Not Either_Ctl_Key_Down
          pe32.Command "[scrollup]"
        End If

        pe32.Command "e"

        If shift = 0 Then               ' If shift = 0_Key_Down
          pe32.Command "[scrollup]"
        End If

        Call Flash(wait_time, num_flash)

      Case &H2500                       ' Left
        If control = 0 Then             ' If Not Either_Ctl_Key_Down
          pe32.Command "[left]"
        End If

        pe32.Command "e"

        If shift = 0 Then               ' If shift = 0_Key_Down
          pe32.Command "[left]"
        End If

        Call Flash(wait_time, num_flash)

      Case &H2700                       ' Right
        If control = 0 Then             ' If Not Either_Ctl_Key_Down
          pe32.Command "[right]"
        End If

        pe32.Command "e"

        If shift = 0 Then               ' If shift = 0_Key_Down
          pe32.Command "[right]"
        End If

        Call Flash(wait_time, num_flash)

      Case &H2300                       ' End  Bottom of files
        pe32.Command "[bottom]"
        pe32.Command "[begin line]"
        pe32.Command "e"
        pe32.Command "[bottom]"
        pe32.Command "[begin line]"
        Call Flash(wait_time, num_flash)

      Case &H2D00                       ' Ins  First file has less lines
        pe32.Command "[scrollup]"
        pe32.Command "e"
        Call Flash(wait_time, num_flash)

      Case &H2E00                       ' Del  First file has more lines
        pe32.Command "[scrolldown]"
        pe32.Command "e"
        Call Flash(wait_time, num_flash)

      Case &H2400                       ' Home Top of files
        pe32.Command "[top]"
        pe32.Command "[begin line]"
        pe32.Command "e"
        pe32.Command "[top]"
        pe32.Command "[begin line]"
        Call Flash(wait_time, num_flash)

      Case &H6B00,43                    ' +
        pe32.Command "[ve GotoDiff] [unmark]"

      Case Else
        pe32.Command "e"                ' Flash toggle files
        Call Flash(wait_time, num_flash)

    End Select

  Loop Until exitloop=1                 ' Repeat loop

  pe32.command Refresh                  ' Restore refresh flag

End Function


'
' Copy mark to cursor position AND move highlighting to new position
'
Function Copy_Mark

  Dim FirstFile

  MType1 = pe32.MarkType                  ' Get mark type

  If MType1 = 0 Then Exit Function        ' Done if no mark anywhere

  MType2 = MType1 And 7                   ' Erase global type

  If MType2 > 3 Then
    MType2 = MType2 - 3                   ' Erase command line flag
  End If

  FirstFile = pe32.Filename               ' File control

  Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

  Call Get_Mark_Size(CommandM, row2, row3, col2, col3)

  Height = row3 - row2
  Width  = col3 - col2

  pe32.Command "[copy mark] [unmark]"     ' Copy the mark

  Select Case MType2
    Case 1                                ' Line mark
      pe32.Command "[down] [mark line]"   ' Start the highlighting
                                          ' *** Above should not be needed if
                                          '   cursor on command line but it is!
      If CommandT = 0 Then
        pe32.Row = row1 + Height + 1
      End If

      pe32.Command "[mark line]"          ' Finish the highlighting

    Case 2                                ' Block mark
      pe32.Command "[mark block]"         ' Start the highlighting

      If CommandT > 0 Then
        pe32.CommandCol = ccol1 + Width
      Else
        pe32.Row = row1 + Height
        pe32.Col = col1 + Width
      End If

      pe32.Command "[mark block]"         ' Finish the highlighting

    Case 3                                ' Character mark
      pe32.Command "[mark char]"          ' Start the highlighting

      If Height = 0 Then
        If CommandT > 0 Then
          pe32.CommandCol = ccol1 + Width  ' If one line block
        Else
          pe32.Col = col1 + Width         ' If one line block
        End If
      Else
        pe32.Row = row1 + Height
        pe32.Col = col3                   ' If multi-line block
      End If

      pe32.Command "[mark char]"          ' Finish the highlighting
  End Select

  Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen state

End Function


'
'   Move cursor to end of line on all files
'
Function End_All

  Call Prompt_Global( "[el]" )

End Function


'
' Execute a FILE of PE32 and/or VBS commands
' Each line is a sequence of commands
'
' Unless a line starts with $ it is assumed to be a set of PE32 commands
'
' Special commands that start with $:
'   $VBScript  VBS-command(s) ' Execute the VBScript commands
'   $ExitIf    VBS-expression ' Exit script if expression is true
'   $Msg       Text           ' Put message on status line
'
' Special commands that start with $ that control execution of following line:
'   $Until     VBS-expression ' Repeat next line until expression true
'   $While     VBS-expression ' Repeat next line while expression true
'
'   $ThenUntil VBS-expression ' Do next line then repeat until expression true
'   $ThenWhile VBS-expression ' Do next line then repeat while expression true
'   $ThenWhileFound           ' Same as $ThenWhile pe32.Found
'
'   $If     VBS-expression    ' Execute next line if expression is true
'   $IfNot  VBS-expression    ' Execute next line if expression is false
'   $Repeat VBS-expression    ' Execute the next line expression times
'
' The above allow one line blocks (multiple commands on a line) and avoids
'          all complications with block marks and nesting.
'
' Special commands that start with $ that set flags for all PE32 lines:
'   $Global                     ' Repeat each line on all files
'   $EndGlobal                  ' Return to executing on current file only
'   $Demo      VBS-expression   ' Zero implies wait for keystroke <> escape key
'                               ' Else expression is wait time in seconds
'   $EndDemo                    ' Same as $Demo <negative value>
'
' Line continuation ("\") is supported on PE32 lines.
'
' This allows unlimited length command sequences with complete documentation
'     and limited control flow.
'
' Example:
'   [ve Ex_Commands("EditTask.cmd")]
'
Function Ex_Commands (FileSpec)

  Dim FSO, TextStream, Command, FullCommand, Special, Rest
  Dim LCSpecial, WhileText, StartFile, ThisFile, FileCount, TestLast

  Set FSO = CreateObject("Scripting.FileSystemObject") ' Setup to read

  ' If empty filename is passed Then use Default.cmd
  If FileSpec = "" Then
    FileSpec = "Default.cmd"
  End If

  ' Verify that passed file exists
  If (FSO.FileExists(FileSpec)) Then

    ' Open passed filename
    Set TextStream = FSO.OpenTextFile(FileSpec, 1) ' Open filename to read

    IsGlobal    = FALSE                            ' Off
    DemoSpeed   = -1                               ' Off
    RepeatCount = 1                                ' Off
    WhileText   = ""                               ' Off

    Do While Not TextStream.AtEndOfStream
      Command = TextStream.ReadLine                ' Get next command

      '
      ' Process the line form the passed file
      '
      If Left(Command, 1) = "$" Then

        ' Remove keyword
        For j=2 To Len(Command)

          If Mid(Command, j, 1) = " " Then         ' Check j'th character
            Exit For
          End If
        Next

        Special = Mid (Command, 2, j-2)            ' Extract special command
        Rest    = Mid (Command, j)                 ' Extract any argument

        LCSpecial = LCase (Special)
        '
        ' Parse special control extensions
        '
        Select Case LCSpecial
          Case "vbscript"
            ExecuteGlobal (Rest)                   ' Execute rest (assume VBScript)

          Case "exitif"
            If Eval(Rest) Then Exit Function

          Case "until"
            WhileText   = "Not " + Rest            ' Save condition
            RepeatCount = 1                        ' Only one of While and Repeat
            TestLast    = 0                        ' Test first

          Case "while"
            WhileText   = Rest                     ' Save condition
            RepeatCount = 1                        ' Only one of While and Repeat
            TestLast    = 0                        ' Test first

          Case "thenuntil"
            WhileText   = "Not " + Rest            ' Save condition
            RepeatCount = 1                        ' Only one of While and Repeat
            TestLast    = 1                        ' Test last

          Case "thenwhile"
            WhileText   = Rest                     ' Save condition
            RepeatCount = 1                        ' Only one of While and Repeat
            TestLast    = 1                        ' Test last

          Case "thenwhilefound"
            WhileText   = "pe32.Found"             ' Save condition
            RepeatCount = 1                        ' Only one of While and Repeat
            TestLast    = 1                        ' Test last

          Case "if"
            If Not Eval(Rest) Then
              If TextStream.AtEndOfStream Then Exit Function
              Command = TextStream.ReadLine        ' Toss next line
            End If

          Case "ifnot"
            If Eval(Rest) Then
              If TextStream.AtEndOfStream Then Exit Function
              Command = TextStream.ReadLine        ' Toss next line
            End If

          Case "repeat"
            RepeatCount = Eval(Rest)
            WhileText   = ""                       ' Only one of While and Repeat

          Case "global"
            IsGlobal = TRUE

          Case "endglobal"
            IsGlobal = FALSE

          Case "demo"
             DemoSpeed = Eval (Rest)               ' Save time

             If DemoSpeed > 10 Then                ' Not too big
               DemoSpeed = 10
             End If

          Case "enddemo"
             DemoSpeed = -1                        ' Off

          Case "msg"
             pe32.Message = Rest                   ' Put Rest on status line

          Case Else
            Answer = MsgBox ("Unknown command: $"+Special+Rest, vbOKCancel)
            If Answer=2 Then Exit Function

        End Select
      Else
        '
        ' Support line continuation
        '
        FullCommand = ""

        Do While Right (Command, 1) = "\" And Not TextStream.AtEndOfStream

          '
          ' Strip lines that start with comment and end with continuation
          ' pe32.Command can not process imbedded comments without error message
          '
          If Left (Command, 2) <> "[*" Then
              FullCommand = FullCommand & Left (Command, Len(Command)-1)
          End If

          Command = TextStream.ReadLine            ' Continue to next line

        Loop

        FullCommand = FullCommand & Command        ' Add part of line without
                                                   '     continuation
        '
        ' Handle any debug requests
        '
        If DemoSpeed = 0 Then
          If FullCommand <> "" And Left(FullCommand, 1) <> "*" Then ' Not dummy
            Answer = MsgBox ("Step "&RepeatCount&": "+FullCommand, vbOKCancel)
            If Answer=2 Then Exit Function
          End If
        End If

        '
        ' If requested, slow it down (badly)
        '
        If DemoSpeed > 0 Then
          If FullCommand <> "" And Left(FullCommand, 1) <> "*" Then ' Not dummy
            StartTime = Timer
            Do
              Elapsed = Timer - StartTime
            Loop Until Elapsed > DemoSpeed
          End If
        End If

        '
        ' Remember on which file we started (and start with next one)
        '
        If IsGlobal Then
           StartFile    = pe32.Filename            ' For global
           pe32.Command "e"                        ' Start with second file
           FileCount    = 0
        End If

        Do_loop = IsGlobal                         ' Save looping status

        '
        ' Loop over files
        '
        Do
          ThisFile = pe32.Filename                 ' Copy current filename
                                                   '     (incase released)
  '       pe32.Found = 1

          If WhileText <> "" Then
            '
            ' Handle While condition
            '
            If TestLast Then
              Do
                pe32.Command FullCommand           ' Execute the PE32 command(s)
              Loop While Eval (WhileText)
            Else
              Do While Eval (WhileText)
                pe32.Command FullCommand           ' Execute the PE32 command(s)
              Loop
            End If
          Else
            '
            ' Handle repeat looping
            '
            If RepeatCount <= 0 Then               ' Validate step counter
              RepeatCount = 1
            End If

            For Ex_Loop = 1 To RepeatCount         ' Support repeat
              pe32.Command FullCommand             ' Execute the PE32 command(s)
            Next
          End If

          '
          ' Handle Global looping and termination
          '
          If IsGlobal Then
            '
            ' Warning:  Due to current PE32 ring ordering this is not failsafe
            '           A new file can be opened and closed resulting in a new
            '           current file and a possible infinite loop.
            '           Also if more files are opened than closed we may never
            '           get back to the first file.
            '
            If StartFile = ThisFile Then           ' Back to original file?
              Do_loop = FALSE                      ' If yes Then stop
            Else
              pe32.Command "e"                     ' Next global file
              FileCount    = FileCount+1           ' Failsafe
            End If
          End If

        ' Exit when processed first file
        Loop Until Do_loop = FALSE Or FileCount > 500

        RepeatCount = 1                            ' Clear count for next command
      End If
    Loop

    TextStream.Close

  Else
    MsgBox FileSpec+" does not exist"
  End If

End Function


'
' Direction:   2 => Last in file              Single file search only
'              1 => Next beyond data cursor
'              0 => This (at data cursor)     Single file search only
'             -1 => Prior before data cursor
'             -2 => First in file             Single file search only
'
' Goto_Book(Direction, Glb)
'
'   Find next bookmark in specified Direction
'   Look in current file unless Glb specified direction to search in multiple files
'
Function Goto_Book(Direction, Glb)

  If Glb <> 0 Then
    ' Find bookmark in multiple files
    Book = Find_Book(Direction)
  Else
    ' Find bookmark in this file
    Book = Get_Book(Direction)
  End If

  If Book > 0 Then
    pe32.command "[bookmark goto '" & Chr(Book) & "']" ' Goto the bookmark
  End If

End Function


'
'
'   Goto next line which is different between current and next file
'   Start compare beyond current line positions
'   Mark the line that is different in the first file
'
Function GotoDiff

  nr1   = pe32.Lines                    ' Get lines and current line in File1
  row1  = pe32.Row

  pe32.Command "e"                      ' File2

  nr2   = pe32.Lines                    ' Get lines and current line in File12
  row2  = pe32.Row

  pe32.Command "e -"                    ' File1

  nrmin = nr1 - row1                    ' Get shortest remaining length
  If nr2 - row2 < nrmin Then
    nrmin = nr2 - row2
  End If

  pe32.Command "[unmark]"               ' Clear last mark

  '   Search current line up to first end of file
  For i=1 To nrmin

    str1=pe32.GetLine(i + row1)

    pe32.Command "e"                    ' File2

    str2=pe32.GetLine(i + row2)

    pe32.Command "e -"                  ' File1

    If str1 <> str2 Then
      pe32.Row = i + row1               ' Move File1

      pe32.Command "e"                  ' File2

      pe32.Row = i + row2               ' Move File2

      pe32.Command "e -"                ' File1

      pe32.Command "[mark line]"        ' Mark File1 and exit
      Exit For
    End If

  Next

End Function


'
'   Parse C error line
'   Pick off filename (and line number if it exists)
'      and open that file (at that line number).
'
Function GotoErr

  Dim regEx, Matches, Match
  Set regEx = New RegExp                ' Use regular expression
  regEx.IgnoreCase = True
  regEx.Global = False

  str=pe32.Line                         ' Line with Filename(line) etc. on it

  '   Find start of line number
  regEx.Pattern = "\("                  ' Set pattern
  Set Matches = regEx.Execute(str)      ' Execute search

  If Matches.count > 0 Then
    For each Match in Matches
      Exit For
    Next

    file = Left(str, Match.FirstIndex)  ' Extract Filename
    pos = Match.FirstIndex+2

    '   Find end of line number
    regEx.Pattern = "\)"
    Set Matches = regEx.Execute(str)

    row = "1"
    If Matches.count > 0 Then
      For each Match in Matches
        Exit For
      Next
      row = Mid(str, pos, Match.FirstIndex) ' Extract line number
    End If
  Else
    file = str                          ' No line number assume just Filename
    row = "1"                           '   at top of file
  End If

  pe32.Command "e " & Chr(34) & file & Chr(34) ' Edit filename
  pe32.Command "line "+row                     '   at line number

End Function


'
'   Move cursor to beginning of line on all files
'
Function Home_All

  Call Prompt_Global( "[bl]" )

End Function


'
'   Place cursor a left edge of field to be left justified.
'   Blanks under the cursor will be moved beyond the following word.
'
'   Apply to rest of file if no marked lines Else apply to rest of mark
'
Function Left_Justify


  ' Get state
  row1  = pe32.Row                      ' Position
  col1  = pe32.Col
  mode1 = pe32.InsertMode               '  and mode

  pe32.InsertMode = 1                   ' Set to insert

  lines1 = pe32.Lines - row1            ' Get lines and current line in File1

  ' Restrict loop to marked lines
  If pe32.MarkType > 0 And pe32.MarkType < 4 Then

    pe32.Command "[begin mark]"         ' Goto start of mark
    row2 = pe32.Row

    If (row2 < row1) Then row2 = row1   ' Use lower of row1 and row2

    pe32.Command "[end mark]"           ' Goto end of mark
    lines1 = pe32.Row - row2            ' Shorten loop to end of mark

    pe32.Row = row2                     ' Restore start position

  End If

  ' Loop over current and rest of lines
  For i=1 To lines1+1                   ' Loop over rest of lines in file

    str = pe32.Line

    char = Mid(str, col1-1, 1)          ' Extract align character

    If char = " " Then

      ' Count moveable blanks
      For j=col1 To Len(str)

        char = Mid(str, j, 1)           ' Extract j'th character

        If char <> " " Then
          Exit For
        End If

      Next

      ' Find end of next word
      For k=j+1 To Len(str)

        char = Mid(str, k, 1)           ' Extract k'th character

        If char = " " Then
          Exit For
        End If

      Next

      ' Move blanks
      '            Before column       next word          blanks moved
      pe32.Line = Left(str,col1-1) + Mid(str,j,k-j) + Mid(str,col1,j-col1) + _
                  Mid(str,k)            ' rest
    End If

    pe32.Row = pe32.Row + 1             ' Get next line
  Next                                  ' Repeat loop

  ' Restore state
  pe32.Row        = row1                ' Restore position
  pe32.Col        = col1
  pe32.InsertMode = mode1               '  and mode

End Function


'
' Move mark to cursor position AND move highlighting to new position
'
Function Move_Mark

  Dim FirstFile

  MType1 = pe32.MarkType                  ' Get mark type

  If MType1 = 0 Then Exit Function        ' Done if no mark anywhere

  MType2 = MType1 And 7                   ' Erase global type

  If MType2 > 3 Then
    MType2 = MType2 - 3                   ' Erase command line flag
  End If

  FirstFile = pe32.Filename               ' File control

  Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

  Call Get_Mark_Size(CommandM, row2, row3, col2, col3)

  Height = row3 - row2
  Width  = col3 - col2

  pe32.Command "[move mark]"              ' Move the mark

  row1  = pe32.Row                        ' Save position (if some of mark is
                                          '   above cursor then cursor moves)
  Select Case MType2
    Case 1                                ' Line mark
      pe32.Command "[down] [mark line]"   ' Start the highlighting

      If CommandT = 0 Then
        pe32.Row = row1 + Height + 1
      End If

      pe32.Command "[mark line]"          ' Finish the highlighting

    Case 2                                ' Block mark
      '
      ' Detect deletion of data to left of cursor
      '
      If CommandT = 0 Then
        If row1 >= row2 And row1 <= row3 And col1 >= col2 Then
          If col1 <= col3 Then
            col1 = col2
          Else
            col1 = col1 - Width - 1
          End If

          pe32.Col = col1
        End If
      Else
        If ccol1 >= col2 Then
          If ccol1 <= col3 Then
            ccol1 = col2
          Else
            ccol1 = ccol1 - Width - 1
          End If

          pe32.CommandCol = ccol1
        End If
      End If

      pe32.Command "[mark block]"         ' Start the highlighting

      If CommandT > 0 Then
        pe32.CommandCol = ccol1 + Width
      Else
        pe32.Row = row1 + Height
        pe32.Col = col1 + Width
      End If

      pe32.Command "[mark block]"         ' Finish the highlighting

    Case 3                                ' Character mark
      '
      ' Detect deletion of data to left of cursor
      '
      If CommandT = 0 Then
        If row1 = row3 And row1 = row2 And col1 >= col2 Then
          If col1 <= col3 Then
            col1 = col2
          Else
            col1 = col1 - Width - 1
          End If

          pe32.Col = col1
        End If
      Else
        If ccol1 >= col2 Then
          If ccol1 <= col3 Then
            ccol1 = col2
          Else
            ccol1 = ccol1 - Width - 1
          End If

          pe32.CommandCol = ccol1
        End If
      End If

      pe32.Command "[mark char]"          ' Start the highlighting

      If row3 = row2 Then
        If CommandT > 0 Then
          pe32.CommandCol = ccol1 + Width ' If one line block
        Else
          pe32.Row = row1 + Height
          pe32.Col = col1 + Width         ' If one line block
        End If
      Else
        pe32.Row = row1 + Height
        pe32.Col = col3                   ' If multi-line block
      End If

      pe32.Command "[mark char]"          ' Finish the highlighting
  End Select

  Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen state

End Function


'
'   Prompt for a string of PE32 commands and then execute then on all open files
'
'   Examples: [ve Prompt_Global] "save" is equivalent to [ve Save_All]
'             [ve Prompt_Global] "[bl]" is equivalent to [ve Home_All]
'             [ve Prompt_Global] "[el]" is equivalent to [ve End_All]
'             [ve Prompt_Global] "[bl][top]" sets up Save_All to work properly
'                                            until true file status is available
'
Function Prompt_Global (Commands)

  Dim FirstFile
  Dim CommandString

  FirstFile = pe32.Filename                 ' Loop control

  If Commands = "" Then
    '
    ' Get input from operator
    '
    CommandString = Inputbox ("Enter Global Command String", "PE32", "")
    '
    ' *** In Win98 user must hit enter or manually task switch back to PE32 **
    ' *** Note however that execution of the script does continue           **
    '
    Call Force_Refresh                      ' Force screen refresh
  Else
    CommandString = Commands
  End If

  If CommandString <> "" Then
    '
    ' Loop over files
    '
    Do
      pe32.Command CommandString            ' Do specified operation
      pe32.Command "e"                      ' Next file

    Loop Until pe32.Filename = FirstFile    ' Exit when back to first file
  End If

End Function


'
'   Add numbers to the line or block mark
'   A start number of 1 and seperator of ") " is assumed unless:
'     If the first line starts with a number then it is used as the start number
'     If the number is followed by a non-blank then it is used as the seperator
'
Function Number_Mark

  MType = pe32.MarkType                   ' Get status of mark

  ' If line or block mark in this file
  If (MType = 1 Or MType = 2) Then

    Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

    pe32.Command "[cursor data] [begin mark]"

    Call Get_Mark_Size(CommandM, row2, row3, col2, col3)

    If MType = 2 Then
      ' Convert block mark to line mark
      pe32.Command "[push mark] [unmark] [mark line]"

      pe32.row = row3
      pe32.Command "[mark line]"
    End If

    Str = pe32.GetMarkedLine(1)

    ' Check for numeric prefix to line
    For i = col2 To Len(Str)
      Char = Asc(Mid(Str, i, 1))          ' Get ith character

      If Char < 48 Or Char > 57 Then Exit For  ' '0' to '9'?
    Next

    ' If we found a number
    If i > col2 Then
      StartNum  = CDbl(Mid(Str,col2,i-col2))

      Seperator = " "                     ' Assume all numeric
      If i > Len(Str) Then
        pe32.SetMarkedLine Left(Str,col2-1), 1 ' Remove the number
      ElseIf i = Len(Str) Then
        If Char = 32 Then
          Seperator = Chr(Char)
        Else
          Seperator = Chr(Char) + " "
        End If

        pe32.SetMarkedLine Left(Str,col2-1), 1 ' Remove the number and seperator
      Else
        If Char <> 32 Then
          Seperator = Chr(Char) + " "

          ' If next char is blank then erase it
          If Asc(Mid(Str, i+1, 1)) = 32 Then i = i+1
        End If

        pe32.SetMarkedLine Left(Str,col2-1) + Mid(Str,i+1), 1 ' Remove the number and seperator
      End If
    Else
      StartNum  = 1
      Seperator = ") "
    End If

    EndNum = StartNum + pe32.MarkedLines - 1

    Digits = Len(EndNum)

    CommandT = pe32.CommandMode           ' Save command mode
    pe32.Command "[cursor data]"

    ' Loop over remainder of block
    For i=1 To pe32.MarkedLines

      Str=pe32.GetMarkedLine(i)

      ' Convert next number to string
      nStr = CStr(StartNum)

      nSize = Len(nStr)

      ' Pad number to largest size and use it to prefix the string
      Str = Left(Str,col2-1) + Space(Digits-nSize) + nStr + Seperator + Mid(Str, col2)

      pe32.SetMarkedLine Str, i

      StartNum = StartNum+1
    Next

    If MType = 2 Then
      pe32.Command "[pop mark]"
    End If

    Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen
  End If

End Function


'
' Open current file in read only mode ... try to position near current position
'
Function Read_Only

  ' Get state
  row1 = pe32.Row                           ' Position
  col1 = pe32.Col

  pe32.Command "view " & Chr(34) & pe32.Filename & Chr(34) ' View current file

  ' Position in same place in read only version
  pe32.Row        = row1                ' Restore position
  pe32.Col        = col1

End Function


'
'   Reflow the area covered by a mark.
'
'   If the area has multiple paragraphs then reflow each separately.
'
'   If Guess_Para is 0 then assumes paragraphs are separated by blank lines
'
'   If Guess_Para is 1 then assumes any line that ends in a "." is also a
'     paragraph (note that this may guess wrong).
'
'   If Guess_Para is 2 then assumes any line that ends in a "." is also a
'     paragraph and adds the missing blank line.
'
Function Reflow_Mult(Guess_Para)

  Reflow_Mult = 0                         ' Set failure

  ' Restrict to marked lines
  If pe32.MarkType > 0 And pe32.MarkType < 4 Then
    CommandT = pe32.CommandMode           ' Save command mode

    row1  = pe32.Row                      ' Save position
    col1  = pe32.Col

    pe32.Command "[begin mark]"           ' Go to top left of block mark

    row2  = pe32.Row                      ' Get top edge of mark

    pe32.Command "[end mark]"             ' Go to lower right of block mark

    row3  = pe32.Row                      ' Get bottom edge of mark

    ' Loop over possible paragraphs
    Do
      pe32.Command "[unmark] [mark line]" ' Mark possible end of paragraph

      flag=2                              ' No end found

      For i=row3 To row2 Step -1
        str=pe32.GetLine(i)               ' Get line

        If str = "" Then
          flag=0                          ' Found separator between paragraphs
          Exit For
        End If

        If Guess_Para > 0 And i < row3 And Right(str,1) = "." Then
          If Guess_Para = 2 Then
            pe32.Row = i
            pe32.Command "[il]"           ' Separate paragraphs

            i = i+1
          End If

          flag=1                          ' Found bottom of prior paragraph
          Exit For
        End If

      Next

      ' Found a paragraph
      If flag > 0 Or i < row3 Then        ' If paragraph not just a blank line
        pe32.Row = i + 1
        pe32.Command "[mark line] [reflow]" ' Mark start of paragraph
      End If

      ' Position above paragraph
      If flag = 0 Then
        row3 = i-1
      Else
        row3 = i
      End If

      pe32.Row = row3                     ' Move to bottom of paragraph
    Loop Until flag = 2 Or i <= row2

    pe32.Command "[unmark]"               ' Clear marks

    pe32.Row = row1                       ' Restore position
    pe32.Col = col1

    Reflow_Mult = 1                       ' Set success
  End If

End Function


'
'   Rem_Book(Start)
'
'   Remove last bookmark at current location beyond Start
'
Function Rem_Book(Start)

  Rem_Book = Get_Book(0)                  ' See if we already have some here

  If Start > Rem_Book Then
    Rem_Book = 0
  End If

  ' If we found one then remove it
  If Rem_Book > 0 Then
    pe32.command "[bookmark remove '" & Chr(Rem_Book) & "']" ' Clear the bookmark
  End If

End Function


'
'   Comment out (add // rem to start of line) all of file
'   Do not double comment lines
'
Function RemAll

  For i=1 To pe32.Lines

    str=pe32.GetLine(i)

    If Left( str, 7 ) <> "// rem " Then
      str="// rem "+str
      pe32.SetLine str, i               ' Add comment
    End If

  Next

End Function


'
'   From current position until matching C block depth
'      comment out (add // rem to start of line) all source lines.
'
'   Note:  Will comment all leading lines before first block mark ({).
'
Function RemFun

  Dim regEx, Matches
  Set regEx = New RegExp                ' Use regular expression
  regEx.IgnoreCase = True               ' Set ignore case
  regEx.Global = True                   ' Set global search

  brace=0
  exitloop=0

  '  Loop over block (or until EOF)
  Do                                    ' Loop until end

    str=pe32.Line                       ' Get pe32 current line

    '   Increment count for blocks opened
    regEx.Pattern = "{"                 ' Set pattern
    Set Matches = regEx.Execute(str)    ' Execute search
    brace=brace+Matches.count           ' Count braces

    ' Decrement count for blocks closed
    regEx.Pattern = "}"                 ' Set pattern
    Set Matches = regEx.Execute(str)    ' Execute search
    brace=brace-Matches.count           ' Count braces

    ' Comment the line
    str="// rem "+str                   ' Add a C remark
    pe32.Line=str                       ' Set current pe32 line
    newrow=pe32.Row+1                   ' Increment row

    ' Exit if EOF
    If newrow>pe32.Lines Then           ' If eof exit
      exitloop=1
    End If

    ' Exit at matching depth
    If brace=0 And Matches.count>0 Then ' If End Function exit
      exitloop=1
    End If

    pe32.Row = newrow                   ' Set new row

  Loop Until exitloop=1                 ' Repeat loop

End Function


'
'   Comment out (add // rem to start of line) marked lines
'   Do not double comment lines
'
Function RemMark

  MType = pe32.MarkType                   ' Get status of mark

  ' If line or block mark in this file
  If (MType = 1 Or MType = 2) Then

    For i=1 To pe32.MarkedLines

      str=pe32.GetMarkedLine(i)

      If Left( str, 7 ) <> "// rem " Then
        str="// rem "+str
        pe32.SetMarkedLine str, i
      End If

    Next

  End If

End Function


'
'   Comment out (with block comments) a single word
'
Function RemWord

  pe32.Word = "/* "+pe32.Word+" */"

End Function


'
'   Place cursor a right edge of field to be right justified.
'   Blanks under the cursor will be moved before the preceding word.
'
'   Apply to rest of file if no marked lines Else apply to rest of mark
'
Function Right_Justify


  ' Get state
  row1  = pe32.Row                      ' Position
  col1  = pe32.Col
  mode1 = pe32.InsertMode               '  and mode

  pe32.InsertMode = 1                   ' Set to insert

  lines1 = pe32.Lines - row1            ' Get lines and current line in File1

  ' Restrict loop to marked lines
  If pe32.MarkType > 0 And pe32.MarkType < 4 Then

    pe32.Command "[begin mark]"         ' Goto start of mark
    row2 = pe32.Row

    If (row2 < row1) Then row2 = row1   ' Use lower of row1 and row2

    pe32.Command "[end mark]"           ' Goto end of mark
    lines1 = pe32.Row - row2            ' Shorten loop to end of mark

    pe32.Row = row2                     ' Restore start position

  End If

  ' Loop over current and rest of lines
  For i=1 To lines1+1                   ' Loop over rest of lines in file

    str = pe32.Line

    If col1 > Len(str) Then

      ' Count moveable blanks
      For j=Len(str) To 1 Step -1

        char = Mid(str, j, 1)           ' Extract j'th character

        If char <> " " Then
          Exit For
        End If

      Next

      ' Find end of next word
      For k=j-1 To 1 step -1

        char = Mid(str, k, 1)           ' Extract k'th character

        If char = " " Then
          Exit For
        End If

      Next

      ' Insert blanks
      pe32.Col = k-1

      For l=1 To col1-j

        pe32.Insert = " "               ' Align word

      Next
    Else
      char = Mid(str, col1+1, 1)        ' Extract align character

      If char = " " Then

        ' Count moveable blanks
        For j=col1 To 1 Step -1

          char = Mid(str, j, 1)         ' Extract j'th character

          If char <> " " Then
            Exit For
          End If

        Next

        If j > 1 Then
          ' Find end of next word
          For k=j-1 To 1 Step -1

            char = Mid(str, k, 1)         ' Extract k'th character

            If char = " " Then
              Exit For
            End If

          Next
        Else
          k=0
        End If

        ' Move blanks
        '           Before word      blanks moved          prior word
        pe32.Line = Left(str,k) + Mid(str,j+1,col1-j) + Mid(str,k+1,j-k) + _
                    Mid(str,col1+1)     ' rest
      End If
    End If

    pe32.Row = pe32.Row + 1             ' Get next line
  Next                                  ' Repeat loop

  ' Restore state
  pe32.Row        = row1                ' Restore position
  pe32.Col        = col1
  pe32.InsertMode = mode1               '  and mode

End Function


'
'   Save all files
'
'   See Save_Changed (above)
'
Function Save_All

  Call Prompt_Global( "save" )

End Function


'
'   Save all changed files
'
Function Save_Changed

  Dim FirstFile

  FirstFile = pe32.Filename               ' Loop control
  '
  ' Loop over files
  '
  Do
    If pe32.Modified > 0 Then             ' Modified?
      pe32.Command "save"                 ' Save the file
    End If

    pe32.Command "e"                      ' Next file

  Loop Until pe32.Filename = FirstFile    ' Exit when back to first file

End Function


'
'   Set_Book(Start)
'
'   Set the next available bookmark beyond Start and all bookmarks at current location
'
'   If no more bookmarks are available then nothing is done (and returns zero)
'
Function Set_Book(Start)

  Have_Book = Get_Book(0)                 ' See if we already have some here

  If Start > Have_Book Then
    Have_Book = Start
  End If

  ' Get bookmark to use (if any)
  Set_Book = New_Book(Have_Book)

  ' If we got one then use it
  If Set_Book > 0 Then
    pe32.command "[bookmark set '" & Chr(Set_Book) & "']" ' Set next bookmark
  End If

End Function


'
'   Set margins to edges of block mark
'
'   Note:  will also do corners (not edges) of character mark ...
'          probably not useful.  Rejects line mark (and some character marks)
'          by rejecting marks one or less characters "wide"
'
Function Set_Margins

  Set_Margins = 0                         ' Set failure

  ' Restrict to marked lines
  If pe32.MarkType > 0 And pe32.MarkType < 4 Then
    CommandT = pe32.CommandMode           ' Save command mode

    row1  = pe32.Row                      ' Save position
    col1  = pe32.Col

    pe32.Command "[begin mark]"           ' Go to top left of block mark

    col2  = pe32.Col                      ' Get left edge of mark

    pe32.Command "[end mark]"             ' Go to lower right of block mark

    col3  = pe32.Col                      ' Get right edge of mark

    pe32.Row        = row1                ' Restore position
    pe32.Col        = col1

    If col3 - col2 > 1 Then
      '
      ' Set margins left right left       ' No first line indent
      '
      pe32.Command "set margins " & col2 & " " & col3 & " " & col2

      Set_Margins = col3-col2+1           ' Set success (return width >= 2)

      pe32.CommandMode = CommandT         ' Restore command mode
    Else
      MsgBox "Bad margins selected:  " & col2 & " " & col3
    End If

  Else
    MsgBox "Select block mark before Set_Margins"
  End If

End Function


'
'   Tab left marked lines by passed number of columns
'
'   Insist cursor is in mark and mark is in data area as a safety measure
'   If no mark or character mark then no action is taken
'
'   If Columns is zero then use length between first two tabs
'
Function Tab_N_Left(Columns)

  Tab_N_Left = 0                          ' Not done

  MType = pe32.MarkType                   ' Get status of mark

  ' If line or block mark in this file and cursor in data
  If (MType = 1 Or MType = 2) And pe32.CommandMode = 0 Then
    SLine = Get_Screen_Line               ' Save scroll position

    row1  = pe32.Row                      ' Save position
    col1  = pe32.Col

    If Columns = 0 Then
      pe32.Command "[begin line] [tab]"   ' Find first tab column
      col2  = pe32.Col                    ' Get that column

      pe32.Command "[tab]"                ' Find next tab column
      col3  = pe32.Col                    ' Get second tab column

      Columns = col3 - col2               ' Length between columns
    End If

    pe32.Command "[begin mark]"           ' Go to top left of block mark

    row2  = pe32.Row                      ' Get top edge of mark
    col2  = pe32.Col

    pe32.Command "[end mark]"             ' Go to lower right of block mark

    row3  = pe32.Row                      ' Get bottom edge of mark
    col3  = pe32.Col

    ' If Cursor is in the mark
    If row1 >= row2 And row1 <= row3 Then

      If MType = 2 Then
        ' Convert block mark to line mark
        pe32.Command "[unmark] [mark line]" ' Mark the bottom
        pe32.Row = row2
        pe32.Command "[mark line]"          ' Mark the top
      End If

      ' See if each line can be shortened
      For i=1 To pe32.MarkedLines

        str=pe32.GetMarkedLine(i)

        If MType = 1 Then
          ' Line mark ... removing from left (verify blanks are there)
          str1 = Left(str, Columns)
        Else
          If col2 <= Columns Then
            Tab_N_Left = 2                  ' Not doable
            Exit For
          End If

          ' Block mark ... removing from left edge (verify blanks are there)
          str1 = Mid(str, col2-Columns, Columns)
        End If

        If Not Is_Blanks(str1) Then
          Tab_N_Left = 2                    ' Not doable
          Exit For
        End If

        pe32.SetMarkedLine str, i
      Next

      If Tab_N_Left = 0 Then                ' If doable
        If MType = 2 Then
          ' Build a string to pad lines with
          Blanks = ""
          For i=1 To Columns
            Blanks = " "+Blanks
          Next
        End If

        ' Shorthen each line
        For i=1 To pe32.MarkedLines

          str=pe32.GetMarkedLine(i)

          If MType = 1 Then
            ' Was a line mark ... remove from left
            If Len(str) >= Columns Then
              str = Right(str, Len(str)-Columns)
            End If
          Else
            ' Was a block mark ... remove from left edge
            If Len(str) >= col2 Then
              If Len(str) >= col3 Then
                str = Left(str, col2 - Columns - 1) & _
                      Mid(str, col2, col3 - col2 + 1) & Blanks & _
                      Right(str, Len(str) - col3)
              Else
                str = Left(str, col2 - Columns - 1) & _
                      Mid(str, col2, col3 - col2 + 1) & Blanks
              End If
            End If
          End If

          pe32.SetMarkedLine str, i
        Next

        Tab_N_Left = 1                       ' Done
      Else
        Columns    = 0
      End If

      If MType = 2 Then
        ' Convert line mark back to block mark
        pe32.Row = row3
        pe32.Col = col3 - Columns
        pe32.Command "[unmark] [mark block]" ' Mark the bottom

        pe32.Row = row2
        pe32.Col = col2 - Columns
        pe32.Command "[mark block]"          ' Mark the top
      End If

      ' Replace cursor
      pe32.Row = row1
      If col1 > Columns Then
        pe32.Col = col1 - Columns
      Else
        pe32.Col = 1
      End If

      Call Set_Screen_Line(Sline)            ' Restore scroll position
    Else
      pe32.Row = row1                        ' Restore position
      pe32.Col = col1

      Call Set_Screen_Line(Sline)            ' Restore scroll position

      pe32.Command "[backtab]"
    End If

  Else
    pe32.Command "[backtab]"
  End If

End Function


'
'   Tab right marked lines in current file by passed number of columns
'
'   Insist data cursor is in mark and mark is in data area as a safety measure
'
'   If no mark or character mark then no action is taken
'
'   If Columns is zero then use length between first two tabs
'
Function Tab_N_Right(Columns)

  Tab_N_Right = 0                         ' Not done

  MType = pe32.MarkType                   ' Get status of mark

  ' If line or block mark in this file and cursor in data
  If (MType = 1 Or MType = 2) And pe32.CommandMode = 0 Then
    SLine = Get_Screen_Line               ' Save scroll position

    row1  = pe32.Row                      ' Save position
    col1  = pe32.Col

    If Columns = 0 Then
      pe32.Command "[begin line] [tab]"   ' Find first tab column
      col2  = pe32.Col                    ' Get that column

      pe32.Command "[tab]"                ' Find next tab column
      col3  = pe32.Col                    ' Get second tab column

      Columns = col3 - col2               ' Length between columns
    End If

    pe32.Command "[begin mark]"           ' Go to top left of block mark

    row2  = pe32.Row                      ' Get top edge of mark
    col2  = pe32.Col

    pe32.Command "[end mark]"             ' Go to lower right of block mark

    row3  = pe32.Row                      ' Get bottom edge of mark
    col3  = pe32.Col

    ' If Cursor is in the mark
    If row1 >= row2 And row1 <= row3 Then

      If MType = 2 Then
        ' Convert block mark to line mark
        pe32.Command "[unmark] [mark line]" ' Mark the bottom
        pe32.Row = row2
        pe32.Command "[mark line]"          ' Mark the top

        ' See if each line can be shortened
        For i=1 To pe32.MarkedLines

          str=pe32.GetMarkedLine(i)

          ' Block mark ... removing from right edge (verify blanks are there)
          str1 = Mid(str, col3 + 1, Columns)

          If Not Is_Blanks(str1) Then
            Tab_N_Right = 2                 ' Not doable
            Exit For
          End If

          pe32.SetMarkedLine str, i
        Next
      End If

      ' Build a string to pad lines with
      Blanks = ""
      For i=1 To Columns
        Blanks = " "+Blanks
      Next

      If Tab_N_Right = 0 Then               ' If doable
        ' Lengthen each line
        For i=1 To pe32.MarkedLines

          str=pe32.GetMarkedLine(i)

          If MType = 1 Then
            ' Was a line mark ... add to beginning of line
            str = Blanks & str
          Else
            ' Was a block mark ... add before left edge
            If Len(str) >= col2 Then
              If Len(str) >= col3 + Columns Then
                str = Left(str, col2-1) & Blanks & _
                      Mid(str, col2, col3 - col2 + 1) & _
                      Right(str, Len(str) - col3 - Columns)
              Else
                str = Left(str, col2-1) & Blanks & _
                      Mid(str, col2, col3 - col2 + 1)
              End If
            End If
          End If

          pe32.SetMarkedLine str, i
        Next
      Else
        Columns    = 0
      End If

      If MType = 2 Then
        ' Convert line mark back to block mark
        pe32.Row = row3
        pe32.Col = col3 + Columns
        pe32.Command "[unmark] [mark block]" ' Mark the bottom

        pe32.Row = row2
        pe32.Col = col2 + Columns
        pe32.Command "[mark block]"          ' Mark the top
      End If

      Tab_N_Right = 1                        ' Done

      ' Replace cursor
      pe32.Row = row1
      pe32.Col = col1 + Columns

      Call Set_Screen_Line(Sline)            ' Restore scroll position
    Else
      pe32.Row = row1                        ' Restore position
      pe32.Col = col1
      Call Set_Screen_Line(Sline)            ' Restore scroll position

      pe32.Command "[tab]"
    End If
  Else
    pe32.Command "[tab]"
  End If

End Function


'
'   Remove numbers from the line or block mark
'   (Removes characters up to and including the first blank)
'
Function UnNumber_Mark

  MType = pe32.MarkType                   ' Get status of mark

  ' If line or block mark in this file
  If (MType = 1 Or MType = 2) Then

    Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

    pe32.Command "[cursor data] [begin mark]"

    Call Get_Mark_Size(CommandM, row2, row3, col2, col3)

    If MType = 2 Then
      ' Convert block mark to line mark
      pe32.Command "[push mark] [unmark] [mark line]"

      pe32.row = row3
      pe32.Command "[mark line]"
    End If

    ' Loop over remainder of block
    For i=1 To pe32.MarkedLines

      Str=pe32.GetMarkedLine(i)

      ' Check for first numeric
      For j = col2 To Len(Str)
        Char = Asc(Mid(Str, j, 1))        ' Get jth character

        If Char < 48 Or Char > 57 Then Exit For  ' '0' to '9'?
      Next

      If j < Len(Str) Then
        ' Check for first blank
        For k = j+1 To Len(Str)
          Char = Asc(Mid(Str, k, 1))        ' Get kth character

          If Char = 32 Then Exit For        ' Blank?
        Next

        ' Convert next number to string
        pe32.SetMarkedLine Mid(Str,1,col2-1)+Mid(Str,k+1), i  ' Remove the number and seperator
      End If
    Next

    If MType = 2 Then
      pe32.Command "[pop mark]"
    End If

    Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen
  End If

End Function



'==============================================================================
'==============================================================================
'
' Utilities generally called from other scripts
'

'
' Direction:   2 => Last in file              Single file search only
'              1 => Next beyond data cursor
'              0 => This (at data cursor)     Single file search only
'             -1 => Prior before data cursor
'             -2 => First in file             Single file search only
'
' Letter = Find_Book(Direction)
'
'   Search in multiple files in the specified direction from cursor for a bookmark
'   Return last match in first file encountered (it will be the new current file)
'   Return zero if no bookmarks are set
'
Function Find_Book(Direction)

  Dim FirstFile

  ' Search current file first
  Find_Book = Get_Book(Direction)

  ' If success then done
  If Find_Book > 0 Then Exit Function

  FirstFile = pe32.Filename               ' Loop control

  If Direction = 1 Then
    pe32.Command "e"                      ' Next file

    If pe32.Filename = FirstFile Then Exit Function ' Exit if only one file
    '
    ' Loop over files
    '
    Do
      Find_Book = Get_Book(-2)            ' Get first

      ' If success then done
      If Find_Book > 0 Then Exit Function

      pe32.Command "e"                    ' Next file

    Loop Until pe32.Filename = FirstFile  ' Exit when back to first file
  End If

  If Direction = -1 Then
    pe32.Command "e -"                    ' Prior file

    If pe32.Filename = FirstFile Then Exit Function ' Exit if only one file
    '
    ' Loop over files
    '
    Do
      Find_Book = Get_Book(2)             ' Get last

      ' If success then done
      If Find_Book > 0 Then Exit Function

      pe32.Command "e -"                  ' Prior file

    Loop Until pe32.Filename = FirstFile  ' Exit when back to first file
  End If

End Function


'
'   Visual flash 2 files
'   (Wait switch back to prior file, wait)
'
'   First argument is amount to wait, second is number of times to flash
'
Function Flash (Wait_Time, Num_Flash)

  If Wait_Time <= 0 Then
    Wait_Time = 100000                    ' Default wait counter
  End If

  If Num_Flash <= 0 Then
    Num_Flash = 1                         ' Set > 1 for multiple flash
  End If

  For k=1 To Num_Flash-1

'   pe32.Command "[wait &Wait_Time&]"
    For i=1 To Wait_Time
      j=j+1
    Next

    pe32.Command "e -"

'   pe32.Command "[wait &Wait_Time&]"
    For i=1 To Wait_Time
      j=j+1
    Next

    pe32.Command "e"

  Next

' pe32.Command "[wait &Wait_Time&]"
  For i=1 To Wait_Time
    j=j+1
  Next

  pe32.Command "e -"

' pe32.Command "[wait &Wait_Time&]"
  For i=1 To Wait_Time
    j=j+1
  Next

End Function


'
'   PE32 may not always update the screen after every script operation.
'   This may be because such updates are turned off (set vbsrefresh off) or
'     because of efficiency reasons.
'
Function Force_Refresh

  Dim Refresh

  Refresh = pe32.Get("vbsrefresh")      ' Save VBS refresh state

  ' Force screen update on
  pe32.command "set vbsrefresh on"

  ' Null operation that triggers screen update
  pe32.Command "[command toggle][command toggle]" ' Force screen refresh

  pe32.command Refresh                  ' Restore refresh flag

End Function


'
' Direction:   2 => Last in file
'              1 => Next beyond data cursor
'              0 => This (at data cursor)
'             -1 => Prior before data cursor
'             -2 => First in file
'
' Letter = Get_Book(Direction)
'
'   Search in current file in the specified direction from cursor for a bookmark
'   Return last bookmark letter found or zero if none is found
'
Function Get_Book(Direction)

  Dim Last_Row,Next_Row,This_Row,Prior_Row,First_Row
  Dim Last_Col,Next_Col,This_Col,Prior_Col,First_Col
  Dim Last_Book,Next_Book,This_Book,Prior_Book,First_Book
  Dim CommandT, ccol1, row1, col1, SLine, col2

  Last_Row = -1
  Next_Row = -1
  This_Row = -1
  Prior_Row = -1
  First_Row = -1

  Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

  pe32.command "[cursor data] [end line] [right 50]" ' Move to unlikely bookmark location

  col2 = pe32.col

  ' Loop over bookmarks "A" to "Z"
  For i = 97 To 122
    pe32.command "[bookmark goto '" & Chr(i) & "']" ' Goto next bookmark

    ' If we moved then we must be at a bookmark
    If pe32.row <> row1 Or (pe32.row = row1 And pe32.col <> col2) Then

      ' First bookmark or previous bookmark was later?
      If First_Row = -1 Or First_Row > pe32.row Or (First_Row = pe32.row And First_Col >= pe32.col) Then
        First_Row  = pe32.row
        First_Col  = pe32.col
        First_Book = i                    ' Save mark letter and position
      End If

      ' First bookmark or previous bookmark was earlier?
      If Last_Row < pe32.row Or (Last_Row = pe32.row And Last_Col <= pe32.col) Then
        Last_Row  = pe32.row
        Last_Col  = pe32.col
        Last_Book = i                     ' Save mark letter and position
      End If

      ' Bookmark is before cursor?
      If pe32.row < row1 Or (pe32.row = row1 And pe32.col < col1) Then

        ' First bookmark or previous bookmark was earlier?
        If Prior_Row < pe32.row Or (Prior_Row = pe32.row And Prior_Col <= pe32.col) Then
          Prior_Row  = pe32.row
          Prior_Col  = pe32.col
          Prior_Book = i                  ' Save mark letter and position
        End If

      ' Bookmark is after cursor?
      ElseIf pe32.row > row1 Or (pe32.row = row1 And pe32.col > col1) Then

        ' First bookmark or previous bookmark was later?
        If Next_Row = -1 Or Next_Row > pe32.row Or (Next_Row = pe32.row And Next_Col >= pe32.col) Then
          Next_Row  = pe32.row
          Next_Col  = pe32.col
          Next_Book = i                   ' Save mark letter and position
        End If

      ' Bookmark is at cursor
      Else
          This_Row  = pe32.row
          This_Col  = pe32.col
          This_Book = i                   ' Save mark letter and position
      End If

      ' Return to known position
      pe32.row = row1
      pe32.col = col2
    End If
  Next

  Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen

  Get_Book = 0                            ' Assume no bookmarks were found
  If Last_Row = -1 Then Exit Function

  ' Select proper return value
  Select Case Direction
    Case 2
      Get_Book = Last_Book

    Case 1
      Get_Book = Next_Book

    Case 0
      Get_Book = This_Book

    Case -1
      Get_Book = Prior_Book

    Case -2
      Get_Book = First_Book

    Case Else
      Answer = MsgBox ("Unknown direction:  "+Direction, vbOKCancel)
  End Select

End Function


'
' Fetch line of screen that has cursor (1 = top edge)
'
Function Get_Screen_Line

  CommandT = pe32.CommandMode             ' Save command mode

  pe32.Command "[cursor data]"            ' Goto data

  row1  = pe32.Row                        ' Save position
  col1  = pe32.Col

  pe32.Command "[top edge]"               ' Top of screen

  Get_Screen_Line = row1 - pe32.Row + 1   ' Compute screen line

  pe32.Row        = row1                  ' Restore position
  pe32.Col        = col1

  pe32.CommandMode = CommandT             ' Restore command mode

End Function


'
'   Return mark size and whether it is on the command line
'
'   If no mark   then StartRow = EndRow = 0
'   If line mark then EndCol = StartCol = 1
'
Function Get_Mark_Size(CommandM, StartRow, EndRow, StartCol, EndCol)

  Dim FirstFile

  MType     = pe32.MarkType               ' Get status of mark

  FirstFile = pe32.Filename               ' Loop control

  CommandM  = 0
  StartRow  = 0
  EndRow    = 0
  StartCol  = 0
  EndCol    = 0

  If MType >= 8 Then                      ' If mark is in another file
    '
    ' Loop over files
    '
    Do
      pe32.Command "e"                    ' Next file

    ' Exit when mark found or back to first file
    Loop Until pe32.MarkType < 8 Or pe32.Filename = FirstFile

    If pe32.Filename = FirstFile Then
      MsgBox "Oops"

      MType = 0
    End If

    MType = MType - 8
  End If

  If MType > 0 Then
    Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

    pe32.Command "[begin mark]"           ' Go to top left of block mark

    If pe32.CommandMode > 0 Then
      CommandM = 1
      StartCol = pe32.CommandCol

      pe32.Command "[end mark]"           ' Go to right edge of mark
      EndCol = pe32.CommandCol

      StartRow = 1
      EndRow   = 1
    Else
      StartRow = pe32.row
      StartCol = pe32.Col

      pe32.Command "[end mark]"           ' Go to lower right of mark
      EndRow = pe32.Row
      EndCol = pe32.Col
    End If

    Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen
  End If

  pe32.Command "e " & Chr(34) & FirstFile & Chr(34) ' Return to first file

End Function


'
' Returns true if String consists of all blanks
'
Function Is_Blanks(String)

  Is_Blanks = True
  '
  ' Loop over characters in String looking for a non-blank one
  '
  For i = 1 to len(String)
    If Mid(String, i, 1) <> " " Then
      Is_Blanks = False                   ' Not all blank
      Exit Function
    End If
  Next

End Function


'
'   Check CURRENT area (current file; data area or command line) for mark type
'
'   Return mark type
'         Return 0 if no          mark
'         Return 1 if line        mark
'         Return 2 if block       mark
'         Return 3 if stream/char mark
'
Function Mark_Type

  Mark_Type = pe32.MarkType           ' Get complete mark type

  If Mark_Type >= 8 Then              ' If in another file then
    Mark_Type = 0                     '   forget it
  End If

  If pe32.CommandMode Then            ' If on command line
    If Mark_Type >= 4 Then            '   and so is mark then
      Mark_Type = Mark_Type-3         '     return mark type
    Else
      Mark_Type = 0                   '   else forget it
    End If
  Else                                ' If in data area
    If Mark_Type >= 4 Then            '   but mark is not then
      Mark_Type = 0                   '     forget it
    End If
  End If

End Function


'
'   Letter = New_Book(Start)
'
'   Get the next available bookmark beyond Start or zero if none are available
'
Function New_Book(Start)

  Call Save_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Save for later restore

  pe32.command "[cursor data] [end line] [right 50]" ' Move to unlikely bookmark location

  col2 = pe32.col

  First = Start + 1
  If First < 97 Then First = 97

  ' Loop over bookmarks "A" to "Z"
  For New_Book = First To 122
    pe32.command "[bookmark goto '" & Chr(New_Book) & "']" ' Goto next bookmark

    If pe32.row = row1 And pe32.col = col2 Then Exit For

    pe32.row = row1                       ' Return to known position
    pe32.col = col2
  Next

  Call Rest_Screen_State(CommandT, ccol1, row1, col1, SLine) ' Restore screen

  If New_Book = 123 Then New_Book=0       ' Change return if all are full

End Function


'
' Restore state of display:
' Command mode, command column, data row, data column, and screen line.
'
Function Rest_Screen_State(CommandT, CommandC, DataR, DataC, SLine)

  pe32.Row         = DataR            ' Restore position
  pe32.Col         = DataC
  pe32.CommandCol  = CommandC

  Call Set_Screen_Line(Sline)         ' Restore scroll position

  pe32.CommandMode = CommandT         ' Restore command mode

End Function


'
' Get current state of display:
' Command mode, command column, data row, data column, and screen line.
'
Function Save_Screen_State(CommandT, CommandC, DataR, DataC, SLine)

  CommandT = pe32.CommandMode         ' Save command mode
  SLine    = Get_Screen_Line          ' Save scroll position

  DataR    = pe32.Row                 ' Save positions
  DataC    = pe32.Col
  CommandC = pe32.CommandCol

' If CommandT Then
'   MsgBox "Active command Cursor at " & CommandC & " and screen cursor at (" & _
'          DataR & "," & DataC & ") on screen row " & SLine
' Else
'   MsgBox "Command Cursor at " & CommandC & " and active screen cursor at (" & _
'          DataR & "," & DataC & ") on screen row " & SLine
' End If

End Function


'
' Restore cursor to passed line of screen (1 = top edge)
'
Function Set_Screen_Line(Line)

  If (Line >= 1) Then
    CommandT = pe32.CommandMode           ' Save command mode

    pe32.Command "[cursor data]"          ' Goto data

    row1  = pe32.Row                      ' Save position
    col1  = pe32.Col

    pe32.Command "[bottom edge]"          ' Bottom of screen

    row3  = pe32.Row                      ' Save position

    pe32.Command "[top edge]"             ' Top of screen

    row2  = pe32.Row                      ' Save position

    If Line <= row3 - row2 + 1 Then       ' If fits on screen
      Line1 = row1 - row2 + 1             ' Compute screen line

      pe32.Row = row2 + Line -1           ' Restore position
      pe32.Col = col1

      If Line1 > Line Then
        pe32.Command "[scrolldown " & Line1-Line & "]" ' Reposition
      End If

      If Line > Line1 Then
        pe32.Command "[scrollup " & Line-Line1 & "]"   ' Reposition
      End If
    Else
      pe32.Row = row1                     ' Restore position
      pe32.Col = col1
    End If

    pe32.CommandMode = CommandT           ' Restore command mode
  End If

End Function



'==============================================================================
'==============================================================================
'
' Documentation and inquiry routines
'

'
'   Capture key code for next keystroke as text (preceded by a blank).
'   Will overwrite current cursor position (or insert at current position
'       if in insert mode).
'
Function Key10

  For i = 1 To 10
    key         = pe32.Key              ' Get next keystroke
    pe32.Insert = " " & key             ' Insert as text

    shift       = pe32.Shift            ' Get shift status for last key
    pe32.Insert = " " & shift           ' Insert as text

    If (shift And   1) > 0 Then pe32.Insert = " Rt Alt"
    If (shift And   2) > 0 Then pe32.Insert = " Lf Alt"
    If (shift And   4) > 0 Then pe32.Insert = " Rt Ctl"
    If (shift And   8) > 0 Then pe32.Insert = " Lf Ctl"
    If (shift And  16) > 0 Then pe32.Insert = " Shift"
    If (shift And  32) > 0 Then pe32.Insert = " NumLk"
    If (shift And  64) > 0 Then pe32.Insert = " ScrlLk"
    If (shift And 128) > 0 Then pe32.Insert = " CapsLk"
    If (shift And 256) > 0 Then pe32.Insert = " Enhncd"

    Call Force_Refresh                  ' Force screen refresh
  Next

End Function


'
'   Capture key code for next keystroke as text (preceded by a blank).
'   Will overwrite current cursor position (or insert at current position
'       if in insert mode).
'
Function KeyCode

  key         = pe32.Key                ' Get next keystroke
  pe32.Insert = " " & key               ' Insert as text

  shift       = pe32.Shift              ' Get shift status for last key
  pe32.Insert = " " & shift             ' Insert as text

End Function
